#!/bin/bash
# Copyright 2018 Badu Networks, Inc.
# HTTP Test Script
revision="1.7"

# Exit codes
#  0 = Success
#  1 = Issue with provided parameters

function usage() {
   echo "Badu Networks HTTP Test Script, version $revision"
   echo "Usage:  $0 --warpHost 192.168.0.1 --warpPort 8000 --standardHost 192.168.0.1 --standardPort 8001 --serverPath /path/to/file [--direction down] [--uploadFile 10k.bin] [--count 20] [--delay 0.2] [--connectTimeout 10] [--maxTime 60]"
   echo -e "\nThis script will perform HTTP transfers using cURL against the provided hosts, alternating between the two. Output is given in CSV format. Results for specified warpHost will be labeled WarpTCP, and results for standardHost will be labeled TCP."
   echo -e "\t--warpHost : IP address or hostname of the optimized path"
   echo -e "\t--warpPort : Destination TCP port for the optimized path"
   echo -e "\t--standardHost : IP address or hostname of the bypass path"
   echo -e "\t--standardPort : Destination TCP port for the bypass path"
   echo -e "\t--serverPath : Path to access on server"
   echo -e "\t--direction [OPTIONAL] : Specify the file transfer direction.  Options are either up or down.  Script defaults to down if this flag is not included."
   echo -e "\t--uploadFile [OPTIONAL] : Required if direction=up. Name of file in current directory to upload to server."
   echo -e "\t--count [OPTIONAL] : Number of iterations. Defaults to 10 if not specified."
   echo -e "\t--delay [OPTIONAL] : Delay between file transfers, in seconds.  Defaults to 0.1 seconds if not specified."
   echo -e "\t--connectTimeout [OPTIONAL] : Specify a custom timeout value (in seconds) for the http connection. Defaults to 15 if not specified."
   echo -e "\t--maxTime [OPTIONAL] : Specify the maximum time allowed (in seconds) for the entire http transfer. Defaults to 30 if not specified."
   echo -e "\t--help : Display this help information"
}


# Set defaults
direction="down"
count=10
delay=0.1
warpLabel="WarpTCP"
stdLabel="TCP"
connectTimeout=15
maxTime=30

# Get parameter values
required=0
while [[ $# -gt 0 ]]
do
   parm="$1"

   case "$parm" in
      --serverPath)
         let required=$required+1
         path="$2"
         shift
         ;;
      --warpHost)
         let required=$required+1
         host1="$2"
         shift
         ;;
      --warpPort)
         let required=$required+1
         port1="$2"
         shift
         ;;
      --standardHost)
         let required=$required+1
         host2="$2"
         shift
         ;;
      --standardPort)
         let required=$required+1
         port2="$2"
         shift
         ;;
      --direction)
         case "$2" in
            up|down) direction="$2" ;;
            *) echo "Direction can only be 'up' or 'down'. Aborting"; exit 1 ;;
         esac
         shift
         ;;
      --uploadFile)
         uploadFile="$2"
         shift
         ;;
      --count)
         count="$2"
         shift
         ;;
      --delay)
         delay="$2"
         shift
         ;;
      --maxTime)
         maxTime="$2"
         shift
         ;;
      --connectTimeout)
         connectTimeout="$2"
         shift
         ;;
      -h|--help)
         usage
         exit 0
         ;;
      *)
         echo "Unknown option: $1"
         echo "Aborting"
         echo "========"
         usage
         exit 1
      ;;
   esac
shift # Next argument or value
done

# Final parameter check
if [ $required -lt 5 ]; then
   echo "Missing parameters, aborting"
   usage;
   exit 1;
elif [ "$direction" == "up" ] && [ "x$uploadFile" == "x" ] ; then
   echo "--uploadFile is required when --direction is set to 'up'"
   usage;
   exit 1;
fi


# Output column headers
echo "Iteration,Direction,Server,Date + Time,TTFB,Total Time,Transfer Speed (bytes/second),Response Code,File Size (bytes),URL" 

# Begin iterating through HTTP transfers
let i=$count-1
while [ $i -ge 0 ];
do
   # Generate timestamp for warp test
   ts=`date "+%Y-%m-%d %H:%M:%S"`

   # Test against warpHost
   if [ "x$direction" == "xup" ]; then
      res1=$i,upload,$warpLabel,`curl --connect-timeout $connectTimeout --max-time $maxTime -w "$ts,%{time_starttransfer},%{time_total},%{speed_upload},%{http_code},%{size_upload},%{url_effective}\n" -o "/dev/null" -F file=@./${uploadFile} -s http://${host1}:${port1}/${path}`
   else
      res1=$i,download,$warpLabel,`curl --connect-timeout $connectTimeout --max-time $maxTime -w "$ts,%{time_starttransfer},%{time_total},%{speed_download},%{http_code},%{size_download},%{url_effective}\n" -o "/dev/null" -s http://${host1}:${port1}/${path}`
   fi

   # Output result and sleep
   echo "$res1"
   sleep $delay

   # Generate timestamp for standard test
   ts=`date "+%Y-%m-%d %H:%M:%S"`

   # Test against standard host
   if [ "x$direction" == "xup" ]; then
      res2=$i,upload,$stdLabel,`curl --connect-timeout $connectTimeout --max-time $maxTime -w "$ts,%{time_starttransfer},%{time_total},%{speed_upload},%{http_code},%{size_upload},%{url_effective}\n" -o "/dev/null" -F file=@./${uploadFile} -s http://${host2}:${port2}/${path}`
   else
      res2=$i,download,$stdLabel,`curl --connect-timeout $connectTimeout --max-time $maxTime -w "$ts,%{time_starttransfer},%{time_total},%{speed_download},%{http_code},%{size_download},%{url_effective}\n" -o "/dev/null" -s http://${host2}:${port2}/${path}`
fi

   # Output result and sleep
   echo "$res2"
   sleep $delay
   let i=$i-1
done

exit 0
